unit GraphXYv02;
// =====================================================================
//
//       2D  
//
// =====================================================================
interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls,
  Dialogs, StdCtrls, ExtCtrls, StrUtils, Math, JPEG;

//      
type TPointXY = record
   X : double;  //   
   Y : double;  //   
end;

//    
type
   TGraphXY = class(TObject)
   private
     //    
     WImg       : TImage;   //  Image
     BevelWImg  : TBevel;   //   Image
     LabelImg   : TLabel;   //   Image
     // ==================================
     fGraphTitul : string;   //   
     // ==================================
     //   WImg
     fWImgColor  : TColor;   //   
     fWImgFont   : TFont ;   //  font-a
     fWImgWidth  : word;     //   Img
     fWImgHeight : word;     //   Img
     // ==================================
     //  
     fNumXCrid   : word;     //     
     fNumYCrid   : word;     //     Y
     fGridColor  : TColor;   //   
     // ==================================
     //   
     fScaleX     : double;   //     X
     fScaleY     : double;   //     Y
     fXMin       : double;   //  fXMin .  
     fXMax       : double;   //  fXMax .  
     fYMin       : double;   //  fYMin .  
     fYMax       : double;   //  fYMax .  
     // ==================================
     //   
     fAxesToZero : boolean;  //     
     fYforAxesX  : double;   // Y -   "X"
     fXforAxesY  : double;   // X -   "Y"
     fAxesColor  : TColor;   //     
     fPicAxesX   : string;   // Format-    X
     fPicAxesY   : string;   // Format-    Y
     // ==================================
     //   
     //      
     procedure CalculateScale();
     //  X  Y      (Left)   (Top)
     function XtoL(X : double): longint;
     function YtoT(Y : double): longint;
     //        (RqLeft, RqTop)
     procedure SubscriptPointLT(RqLeft, RqTop : longint; RqColor : TColor; RqText : string);
     //     ( RqY)
     //    ( RqX) 
     procedure ShowHLine(RqY : double; RqColor : TColor; RqText : string);
     procedure ShowVLine(RqX : double; RqColor : TColor; RqText : string);
     //    
     procedure EraseGraphXY();
     //   
     procedure ShowGridLine();
     //    GraphXY
     procedure RepaintGraphXY();
     // ==================================
     //    
     procedure SetGraphTitul (GraphTitul : string);
     procedure SetImgBackGroundColor (BackGroundColor : TColor);
     procedure SetGridColor (GridColor : TColor);
     procedure SetAxesColor (AxesColor : TColor);
     procedure SetPicAxesX (PicAxesX : string);
     procedure SetPicAxesY (PicAxesY : string);
   public
     // ==================================
     //  
     // ==================================
     property GraphTitul : string read fGraphTitul write SetGraphTitul;
     property BackGroundColor : TColor read fWImgColor write SetImgBackGroundColor;
     property GridColor : TColor read fGridColor write SetGridColor;
     property AxesColor : TColor read fAxesColor write SetAxesColor;
     property PicAxesX : string read fPicAxesX write SetPicAxesX;
     property PicAxesY : string read fPicAxesY write SetPicAxesY;
     property ReadScaleX : double read fScaleX;
     property ReadScaleY : double read fScaleY;
     // ==================================
     //   
     // ==================================
     //   
     procedure OpenGraphXY(RqPanel: TPanel; UniId : string);
     function  CloseGraphXY(RqGraphXY : TGraphXY) : TGraphXY;
     // ==================================
     //   
     // ==================================
     //   /   X  Y,  
     //        
     procedure SetScaleXY (RqScaleX,  RqScaleY : double);
     //       
     procedure SetMinMaxXY (RqXMin, RqXMax, RqYMin, RqYMax: double);
     //        
     procedure FindOptimalScale(RqXY : array of TPointXY);
     //      
     procedure ShowGraphXY (RqXY : array of TPointXY; RqColor : TColor);
     //       (RqX, RqY)
     procedure SubscriptPointXY(RqX, RqY : double; RqColor : TColor; RqText : string);
     //   (  )      (RqX, RqY)
     procedure PaintPointXY(RqX, RqY : double;
                            RqLnColor, RqBkColor : TColor; RqRadius : byte);
     //    X Y -    
     procedure VectotToGraphXY(XB, YB, XE, YE : double; RqColor : TColor);
     //      
     procedure FullEraseGraphXY();
     //     . (RqGFormat: 'J' - jpg   'B' -bmp)
     procedure SaveGraphXY(RqFullFileName : string; RqGFormat : char);
end;

// ===================================================================================
// ===================================================================================
implementation
// ===================================================================================
// ===================================================================================

const
   //  WImg   
   PBordUp    = 26;
   PBordDown  = 6;
   PBordLeft  = 6;
   PBordRight = 6;
   PBordHeight  = PBordUp + PBordDown;
   PBordWidth   = PBordLeft + PBordRight;
   //   WImg
   IBordUp      = 8;
   IBordDown    = 8;
   IBordLeft    = 8;
   IBordRight   = 8;
   IBordHeight  = IBordUp + IBordDown + 1;
   IBordWidth   = IBordLeft + IBordRight + 1;
   //   WImg
   MinWidth     = 50 + IBordWidth;
   MinHeight    = 30 + IBordHeight;

// ======================================================
//
//     GraphXY
//
// ======================================================
//      
procedure TGraphXY.CalculateScale();
begin
   if Abs(fXMax-fXMin) > 0
   then begin
       //    
       fScaleX:= fWImgWidth /Abs(fXMax-fXMin);
   end
   else begin
       ShowMessage('GraphXY :       X');
   end;
   if Abs(fYMax-fYMin) > 0
   then begin
       //    
       fScaleY:= fWImgHeight /Abs(fYMax-fYMin);
   end
   else begin
       ShowMessage('GraphXY :       Y');
   end;
end;

//  X      (Left)
function TGraphXY.XtoL(X : double): longint;
begin
 Result:= 0;
 if fScaleX > 0
 then begin
    Result := Round(fScaleX*(X-fXMin)) + IBordLeft;
 end;
end;

//  Y      (Top)
function TGraphXY.YtoT(Y : double): longint;
begin
 Result:= 0;
 if fScaleY > 0
 then begin
    //  
    Result := fWImgHeight - Round(fScaleY*(Y-fYMin) - IBordUp);
 end;
end;

//        (RqLeft, RqTop)
procedure TGraphXY.SubscriptPointLT(RqLeft, RqTop : longint; RqColor : TColor; RqText : string);
const LOffs = 2;
      TOffs = 2;
var   SaveColor  : TColor;
begin
  with WImg.Canvas do
  begin
     if RqText <> ''
     then begin
       //   
       if (RqLeft + TextWidth(RqText) + LOffs <= WImg.Width) and
          (RqTop + TextHeight(RqText) + TOffs <= WImg.Height)
       then begin
          //      WImg
          SaveColor  := Font.Color;
          Font.Color := RqColor;
          //  
          TextOut(RqLeft + LOffs, RqTop + TOffs, RqText);
          Font.Color := SaveColor;
       end;
     end;
  end;  {of with WImg.Canvas}
end; {of procedure SubscriptPoint}


//     
procedure TGraphXY.ShowHLine(RqY : double; RqColor : TColor; RqText : string);
const LOffs = 4;
      TOffs = 2;
var   WPenL, WPenT : longint;
      SaveColor    : TColor;
begin
  with WImg.Canvas do
  begin
    //      Img
    SaveColor := Pen.Color;
    Pen.Color := RqColor;
    WPenL:= 0;
    WPenT:= YtoT(RqY);
    MoveTo(WPenL,WPenT);
    WPenL:= WImg.Width;
    LineTo(WPenL,WPenT);
    Pen.Color := SaveColor;
    //  
    if RqText <> ''
    then begin
      WPenL:= XtoL (fXforAxesY);  // X -   "Y"
      if (WPenL +  TextWidth(RqText) + LOffs) >  WImg.Width
      then begin
         //       Y
         WPenL := WPenL -  TextWidth(RqText) - LOffs;
      end
      else begin
         //       Y
         WPenL := WPenL + LOffs;
      end;
      WPenT:= YtoT(RqY);
      if fPicAxesY <> ''
      then SubscriptPointLT(WPenL, WPenT, clBlack, RqText);
    end;
  end; {of with WImg.Canvas}
end; {procedure ShowHLine}

//      
procedure TGraphXY.ShowVLine(RqX : double; RqColor : TColor; RqText : string);
const LOffs = 2;
      TOffs = 2;
var   WPenL, WPenT : longint;
      SaveColor    : TColor;
begin
  with WImg.Canvas do
  begin
     //  
     SaveColor := Pen.Color;
     Pen.Color := RqColor;
     WPenT:= 0;
     WPenL:= XtoL(RqX);
     MoveTo(WPenL,WPenT);
     WPenT:= WImg.Height;
     LineTo(WPenL,WPenT);
     Pen.Color := SaveColor;
     //  
     if RqText <> ''
     then begin
        WPenT:= YtoT (fYforAxesX);  // Y -   "X"
        if (WPenT +  TextHeight(RqText) + TOffs) >  WImg.Height
        then begin
           //     
           WPenT := WPenT - TextHeight(RqText) - TOffs;
        end
        else begin
           //     
           WPenT := WPenT + TOffs;
        end;
        WPenL:= XtoL(RqX);
        if fPicAxesX <> ''
        then SubscriptPointLT(WPenL, WPenT, clBlack, RqText);
     end;
  end;  {of with WImg.Canvas}
end; {of procedure ShowVLine}

//    
procedure TGraphXY.EraseGraphXY();
begin
  with WImg.Canvas do
  begin
     Brush.Color := fWImgColor;
     FillRect(Rect(0,0, WImg.Width, WImg.Height ));
  end;
end; {of procedure EraseGraphXY}


//      
procedure TGraphXY.ShowGridLine();
var SStep, SNext  : double;
var EnYAxesOverZeroX, EnXAxesOverZeroY : boolean;
var WStr : string;
begin
  with WImg.Canvas do
  begin
        //     
        EnXAxesOverZeroY := False;
        EnYAxesOverZeroX := False;
        //   Y   "X"
        if (fYMax > 0) and (fYMin < 0)
        then begin
           EnXAxesOverZeroY := True;       //  X    Y = 0
           fYforAxesX    :=0;              // Y -   "X"
        end
        else begin
           if (fYMin >= 0) and (fYMax > 0)
           then fYforAxesX := fYMin;       // (Max  Min > 0
           if (fYMin < 0) and (fYMax <= 0)
           then fYforAxesX := fYMax;       // (Max  Min < 0
        end;
        //   X   "Y"
        if (fXMin < 0) and (fXMax > 0)
        then begin
           EnYAxesOverZeroX := True;       //  Y    X = 0
           fXforAxesY    := 0;             // X -   "Y"
        end
        else begin
           if (fXMin >= 0) and (fXMax > 0)
           then fXforAxesY := fXMin;       // (Max  Min > 0
           if (fXMin < 0) and (fXMax <= 0)
           then fXforAxesY := fXMax;       // (Max  Min < 0
        end;
        // ======================
        // -X      
        SStep := (fXMax - fXMin) / fNumXCrid;  //   X
        if (not EnYAxesOverZeroX)
        then begin
            //      fXMin  fXMax
            SNext := fXMin;
            repeat
               WStr:= Format(fPicAxesX, [SNext]);
               ShowVLine(SNext, fGridColor, WStr);
               SNext := SNext + SStep;
            until (SNext > fXMax);
            //    "Y"
            ShowVLine(fXforAxesY, fAxesColor, '');
        end
        else begin
           //      0  fXMax
           SNext := 0;
           repeat
              WStr:= Format(fPicAxesX, [SNext]);
              ShowVLine(SNext, fGridColor, WStr);
              SNext := SNext + SStep;
           until (SNext > fXMax);
           //      - SStep  fXMin
           SNext := - SStep;
           repeat
              WStr:= Format(fPicAxesX, [SNext]);
              ShowVLine(SNext, fGridColor, WStr);
              SNext := SNext - SStep;
           until (SNext < fXMin);
           //    "Y"
            ShowVLine(fXforAxesY, fAxesColor, '');
        end;
        // ======================
        // -Y      
        SStep := (fYMax - fYMin) / fNumYCrid;   //   Y
        if (not EnXAxesOverZeroY)
        then begin
            //      fYMin  fYMax
            SNext := fYMin;
            repeat
               if SNext <> fYforAxesX
               then begin
                  WStr:= Format(fPicAxesY, [SNext]);
                  ShowHLine(SNext, fGridColor, WStr);
               end;
               SNext := SNext + SStep;
            until (SNext > fYMax);
            //    "X"
            ShowHLine(fYforAxesX, fAxesColor, '');
        end
        else begin
           //      SStep  fYMax
           SNext := SStep;
           repeat
              WStr:= Format(fPicAxesY, [SNext]);
              ShowHLine(SNext, fGridColor, WStr);
              SNext := SNext + SStep;
           until (SNext > fYMax);
           //      SStep  fYMin
           SNext := - SStep;
           repeat
              WStr:= Format(fPicAxesY, [SNext]);
              ShowHLine(SNext, fGridColor, WStr);
              SNext := SNext - SStep;
           until (SNext < fYMin);
           //    "X"
            ShowHLine(fYforAxesX, fAxesColor, '');
        end;
  end; {of with WImg.Canvas}
end; {of procedure ShowGridLine}

//    GraphXY
procedure TGraphXY.RepaintGraphXY();
begin
  EraseGraphXY();
  if (fScaleX > 0) and (fScaleY > 0)
  then ShowGridLine();
end; {}

// ======================================================
//
//    GraphXY
//
// ======================================================
procedure TGraphXY.SetGraphTitul (GraphTitul : string);
begin
  fGraphTitul := GraphTitul;
  LabelImg.Caption := fGraphTitul;
end;
procedure TGraphXY.SetImgBackGroundColor (BackGroundColor : TColor);
begin
   fWImgColor := BackGroundColor;
   RepaintGraphXY();
end;
procedure TGraphXY.SetGridColor (GridColor : TColor);
begin
   fGridColor := GridColor;
   RepaintGraphXY();
end;
procedure TGraphXY.SetAxesColor (AxesColor : TColor);
begin
   fAxesColor := AxesColor;
   RepaintGraphXY();
end;
procedure TGraphXY.SetPicAxesX (PicAxesX : string);
begin
   fPicAxesX := PicAxesX;
   RepaintGraphXY();
end;
procedure TGraphXY.SetPicAxesY (PicAxesY : string);
begin
   fPicAxesY := PicAxesY;
   RepaintGraphXY();
end;
// ======================================================
//
//    GraphXY
//
// =======================================================

//   /   X  Y,  
//        
procedure TGraphXY.SetScaleXY (RqScaleX,  RqScaleY : double);
begin
  if (RqScaleX > 0) and (RqScaleY > 0)
  then begin
      fScaleX := RqScaleX;
      fScaleY := RqScaleY;
      fXMax :=  (fWImgWidth / 2) / fScaleX;
      fXMin := - fXMax;
      fYMax :=  (fWImgHeight / 2) / fScaleY;
      fYMin := - fYMax;
      RepaintGraphXY();
  end;
end; {of procedure SetScaleXY}

//       
procedure TGraphXY.SetMinMaxXY (RqXMin, RqXMax, RqYMin, RqYMax: double);
begin
  if (RqXMin < RqXMax) and (RqYMin < RqYMax)
  then begin
      fXMin := RqXMin;
      fXMax := RqXMax;
      fYMin := RqYMin;
      fYMax := RqYMax;
      CalculateScale();
      RepaintGraphXY();
  end
  else begin
       ShowMessage('GraphXY : M      !');
  end;
end; {of procedure SetMinMaxXY}

// 11.10.2008 (   1,2)
//        
procedure TGraphXY.FindOptimalScale(RqXY : array of TPointXY);
var IndB, IndE : integer;
var Ind : integer;
begin
  if (Addr(RqXY) <> nil)
  then begin
    //       
    fScaleX:=0;
    fScaleY:=0;
    try
    //   
    // ============================
       if (Low(RqXY) < High(RqXY))
       then begin
          //   , 
          //       
          IndB := Low(RqXY);
          IndE := High(RqXY);
          fXMin := RqXY[IndB].X;
          fXMax := fXMin;
          fYMin := RqXY[IndB].Y;
          fYMax := fYMin;
          for Ind := IndB to IndE do
          begin
             with RqXY[Ind] do
             begin
                if X < fXMin then fXMin := X;
                if X > fXMax then fXMax := X;
                if Y < fYMin then fYMin := Y;
                if Y > fYMax then fYMax := Y;
             end;
          end;
          //    Y   
          if (fYMax = fYMin)
          then begin
            if (fYMax = 0)
            then begin
               //     
               // (  )
               fYMin := -0.1;
               fYMax :=  0.1;
            end
            else begin
                if (fYMax > 0)
                then fYMin := 0 else fYMax := 0;
            end;
          end;
          //       .
          //  ,   
          //        .
          if (fXMax = fXMin)
          then begin
            if (fXMax = 0)
            then begin
               //     
               // (  )
               fXMin := -0.1;
               fXMax :=  0.1;
            end   
            else begin
               if (fXMax > 0)
               then fXMin := 0 else fXMax := 0;
            end;
          end;
          //     
          CalculateScale();
          RepaintGraphXY();
       end;
    // ============================
    //   
    except
       fXMin := 0;
       fXMax := 0;
       fYMin := 0;
       fYMax := 0;
       ShowMessage('TGraphXY.FindOptimalScale :' +
                   '   . ');
    end;
  end;
end; {of procedure FindOptimalScale}


//      
procedure TGraphXY.ShowGraphXY
          (RqXY : array of TPointXY; RqColor : TColor);
var IndB, IndE : integer;
var Ind : integer;
var WPenX, WPenY : longint;
begin
 if (Addr(RqXY) <> nil)
 then begin
    //  , 
    //     
    if not ((fScaleX > 0) and (fScaleY > 0))
    then FindOptimalScale(RqXY);
    //       
    //    
    if (fScaleX > 0) and (fScaleY > 0)
    then begin
      try
      //   
      // ============================
         if (Low(RqXY) < High(RqXY))
         then begin
            IndB := Low(RqXY);
            IndE := High(RqXY);
            //  
            with WImg.Canvas do
            begin
               Pen.Color := RqColor;
               WPenX:= XtoL(RqXY[IndB].X);
               WPenY:= YtoT(RqXY[IndB].Y);
               MoveTo(WPenX,WPenY);
               for Ind := IndB to IndE do
               begin
                  with RqXY[Ind] do
                  begin
                     WPenX:= XtoL(X);
                     WPenY:= YtoT(Y);
                     LineTo(WPenX,WPenY);
                  end; {of with RqXY[Ind]}
               end; {of for Ind := IndB}
            end; {of with WImg.Canvas}
         end; {if (Low(RqXY) < High(RqXY))}
      // ============================
      //   
      except
         ShowMessage('TGraphXY.FindOptimalScale :   . ');
      end;
      // ============================
    end;
 end;
end; {of procedure ShowGraphXY}

//        (RqX, RqY)
procedure TGraphXY.SubscriptPointXY(RqX, RqY : double; RqColor : TColor; RqText : string);
var   WLeft, WTop : longint;
begin
  if (fScaleX > 0) and (fScaleY > 0)
  then begin
     WLeft:= XtoL(RqX);
     WTop:= YtoT(RqY);
     SubscriptPointLT(WLeft, WTop, RqColor, RqText);
  end
  else begin
     ShowMessage('TGraphXY.SubscriptPointXY :    ');
  end;
end; {of procedure SubscriptPoint}

// 10.10.2008
//   (  )      (RqX, RqY)
procedure TGraphXY.PaintPointXY(RqX, RqY : double;
                                RqLnColor, RqBkColor : TColor; RqRadius : byte);
var   WLeft, WTop   : longint;
      SaveStyle     : TBrushStyle;
      SaveLnColor   : TColor;  //  Brush.Color
      SaveBkColor   : TColor;  //  Pen.Color
begin
  if (fScaleX > 0) and (fScaleY > 0) and (RqRadius > 0)
  then begin
     //      
     WLeft := XtoL(RqX);
     WTop  := YtoT(RqY);
     //  
     with WImg.Canvas do
     begin
        //   
        SaveStyle     := Brush.Style;
        SaveBkColor   := Brush.Color;
        SaveLnColor   := Pen.Color;
        //    
        Brush.Color   := RqBkColor;
        Brush.Style   := bsSolid;
        Pen.Color     := RqLnColor;
        //  
        Ellipse(WLeft - RqRadius, WTop - RqRadius, WLeft + RqRadius, WTop + RqRadius);
        //   
        Brush.Style := SaveStyle;
        Brush.Color := SaveBkColor;
        Pen.Color   := SaveLnColor;
     end;
  end
  else begin
     ShowMessage('TGraphXY.PaintPointXY :    ');
  end;
end; {of procedure PaintPointXY}

//    X Y -    
procedure TGraphXY.VectotToGraphXY(XB, YB, XE, YE : double; RqColor : TColor);
var WPenX, WPenY : longint;
begin
   if (fScaleX > 0) and (fScaleY > 0)
   then begin
      with WImg.Canvas do
      begin
         Pen.Color := RqColor;
         WPenX:= XtoL(XB);
         WPenY:= YtoT(YB);
         MoveTo(WPenX,WPenY);
         WPenX:= XtoL(XE);
         WPenY:= YtoT(YE);
         LineTo(WPenX,WPenY);
      end; {of with WImg.Canvas}
      PaintPointXY(XE, YE, RqColor, RqColor, 2);
   end;
end; {of procedure VectotToGraphXY}

//      
procedure TGraphXY.FullEraseGraphXY();
begin
    //   
    fScaleX:=0;
    fScaleY:=0;
    fXMin := 0;
    fXMax := 0;
    fYMin := 0;
    fYMax := 0;
    RepaintGraphXY();
end; {of procedure FullEraseGraphXY}


//     . (RqGFormat: 'J' - jpg   'B' -bmp)
procedure TGraphXY.SaveGraphXY(RqFullFileName : string; RqGFormat : char);
var  wPath, wName, wExt, wFileName : string;
     wPPos  : word;
     wOK    : boolean;
var  Jpeg   : TJPEGImage;  //   unit jpeg.
     BitMap : TBitMap;
begin
   wOK   := False;
   wPath := ExtractFilePath(RqFullFileName);  //  'Disk:\dir1\..\DirN\'
   wName := ExtractFileName(RqFullFileName);  //  'Name.Ext'  ''
   WExt  := ExtractFileExt(wName);            //  '.Ext'  ''
   if Length(WExt) > 0
   then begin
       //   
       wPPos := Pos('.', wName);
       if (wPPos > 1)
       then begin
            wName := LeftStr(wName,wPPos-1);
            wFileName := wPath + wName;
            wOK := True;
       end;
   end
   else begin
       //     ( )
       wFileName := wPath + wName;
       wOK := True;
   end;
   if WOk
   then begin
       case UpCase(RqGFormat) of
       // ==========================
       'J' : begin      //   Jpeg
               wFileName := wFileName + '.jpg';
               Jpeg := TJPEGImage.Create;
               try
                  with Jpeg do
                  begin            //  Bitmap  JPG
                     Assign(WImg.Picture.Bitmap);
                     SaveToFile(wFileName);
                  end;
               finally
                     Jpeg.Free;    //   JPG
               end;
             end;
          // ==========================
         'B': begin       //   BMP
                wFileName := wFileName + '.bmp';
                BitMap := TBitMap.Create;
                try
                   with BitMap do   //  Bitmap  BMP
                   begin
                      Assign(WImg.Picture.Bitmap);
                      SaveToFile(wFileName);
                   end;
                finally
                      BitMap.Free;  //   BMP
                end;
              end;
       end; {of case}
   end;
end; {of procedure SaveGraphXY}

// ======================================================
//
//      GraphXY
//
// ======================================================
procedure TGraphXY.OpenGraphXY(RqPanel: TPanel; UniId : string);
begin
  if (RqPanel.Width > (PBordLeft + PBordRight + MinWidth)) and
     (RqPanel.Height > (PBordUp + PBordDown + MinHeight))
  then begin
      //   
      // =============================
      //   
      //   Image
      WImg := TImage.Create(RqPanel);
      WImg.Parent := RqPanel;
      with WImg do
      begin
        Name    := 'Img' + UniId;
        Top     := PBordUp;
        Left    := PBordLeft;
        Width   := RqPanel.Width - PBordWidth;
        Height  := RqPanel.Height - PBordHeight;
        Visible := True;
        //    
        fWImgWidth  := Width - IBordWidth;   //   Img
        fWImgHeight := Height - IBordHeight; //   Img
      end;
      //     Image
      BevelWImg  := TBevel.Create(RqPanel);
      BevelWImg.Parent := RqPanel;
      with BevelWImg do
      begin
         Name    := 'Bevel' + UniId;
         Shape   := bsBox;
         Top     := PBordUp;
         Left    := PBordLeft;
         Width   := RqPanel.Width - PBordWidth;
         Height  := RqPanel.Height - PBordHeight;
         Visible := True;
      end;
      //     Image
      LabelImg   := TLabel.Create(RqPanel);
      LabelImg.Parent := RqPanel;
      with LabelImg do
      begin
         Name     := 'Label' + UniId;
         Top      := 5;
         Left     := PBordLeft;
         AutoSize := True;
         Caption  := '    ';
         Visible  := True;
      end;
      //  
      fScaleX :=0;
      fScaleY :=0;

      //   
      // =====================
      //        
      fWImgColor  := RqPanel.Color;
      fWImgFont   := RqPanel.Font;
      // =====================
      fAxesToZero := True;              //     
      fAxesColor  := clBlack;           //     
      fPicAxesX   := '%-0.2f';          // Format-    X
      fPicAxesY   := '%-0.2f';          // Format-    Y
      // =====================
      fNumXCrid   := 10;                //     
      fNumYCrid   := 10;                //     Y
      fGridColor  := RGB(128,128,128);  //   
      // =====================
      EraseGraphXY();                  //   


  end;
end;

function TGraphXY.CloseGraphXY(RqGraphXY : TGraphXY) : TGraphXY;
var Parent : TWinControl;
begin
   Result := RqGraphXY;
   if RqGraphXY <> nil
   then begin
      Parent := RqGraphXY.WImg.Parent;
      with RqGraphXY do
      begin
         //   
         LabelImg.Free;
         BevelWImg.Free;
         WImg.Free;
      end;
      //   
      RqGraphXY.Free;
      Parent.Repaint;
      Result := nil;
   end;
end;



end.
